/**
 * BOLILLERO - Lógica de Sorteo
 * Comunicación con el backend y control del sorteo
 */

// Variables globales del sorteo
let sorteoActivo = false;
let sorteoData = null;
let indexActual = 0;
let intervaloSorteo = null;

/**
 * Iniciar sorteo
 */
function iniciarSorteo(config) {
    if (sorteoActivo) {
        console.warn('Ya hay un sorteo en progreso');
        return;
    }
    
    sorteoActivo = true;
    indexActual = 0;
    
    // Mostrar modal de progreso
    mostrarModalSorteo();
    
    // Realizar petición AJAX al backend
    fetch('../ajax/ejecutar_sorteo.php', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json'
        },
        body: JSON.stringify({
            desde: config.desde,
            hasta: config.hasta,
            repetir: config.repetir
        })
    })
    .then(response => response.json())
    .then(data => {
        if (data.error) {
            alert('Error: ' + data.mensaje);
            cerrarModalSorteo();
            sorteoActivo = false;
            return;
        }
        
        sorteoData = data;
        
        // Iniciar animación del sorteo
        reproducirSorteo();
    })
    .catch(error => {
        console.error('Error:', error);
        alert('Error al ejecutar el sorteo');
        cerrarModalSorteo();
        sorteoActivo = false;
    });
}

/**
 * Reproducir sorteo con animaciones
 */
function reproducirSorteo() {
    if (!sorteoData || !sorteoData.sorteos) {
        return;
    }
    
    const sorteos = sorteoData.sorteos;
    const velocidad = CONFIG.velocidad || 1.8;
    const tiempoBase = 800; // ms
    const tiempoSorteo = tiempoBase / velocidad;
    
    // Elementos del modal
    const numeroSorteo = document.getElementById('numeroSorteo');
    const sorteoActualSpan = document.getElementById('sorteoActual');
    const aparicionesActuales = document.getElementById('aparicionesActuales');
    const aparicionesTotal = document.getElementById('aparicionesTotal');
    const infoGanador = document.getElementById('infoGanador');
    const progresoFill = document.getElementById('progresoFill');
    const porcentajeConDos = document.getElementById('porcentajeConDos');
    const bolillaSorteo = document.getElementById('bolillaSorteo');
    
    // Configurar información inicial
    infoGanador.textContent = sorteoData.ganador;
    aparicionesTotal.textContent = sorteos[sorteos.length - 1].contador_ganador;
    
    // Función para mostrar cada sorteo
    function mostrarSorteo(index) {
        if (index >= sorteos.length) {
            // Sorteo completado
            finalizarSorteo();
            return;
        }
        
        const sorteo = sorteos[index];
        
        // Actualizar número
        numeroSorteo.textContent = sorteo.numero;
        sorteoActualSpan.textContent = sorteo.sorteo;
        aparicionesActuales.textContent = sorteo.contador_ganador;
        
        // Actualizar progreso
        const progreso = (index / sorteos.length) * 100;
        progresoFill.style.width = progreso + '%';
        
        // Calcular estadísticas simuladas
        const porcentaje = Math.min(100, Math.round((index / sorteos.length) * 100));
        porcentajeConDos.textContent = porcentaje;
        
        // Efecto visual cuando sale el ganador
        if (sorteo.es_ganador) {
            bolillaSorteo.style.boxShadow = '0 0 40px rgba(255, 215, 0, 0.8)';
            numeroSorteo.style.color = '#FFD700';
            
            // Vibración si está disponible
            if (navigator.vibrate) {
                navigator.vibrate([50, 100, 50]);
            }
            
            setTimeout(() => {
                bolillaSorteo.style.boxShadow = '';
                numeroSorteo.style.color = '';
            }, 500);
        }
        
        indexActual = index + 1;
    }
    
    // Iniciar intervalo
    intervaloSorteo = setInterval(() => {
        mostrarSorteo(indexActual);
    }, tiempoSorteo);
}

/**
 * Finalizar sorteo y mostrar resultado
 */
function finalizarSorteo() {
    clearInterval(intervaloSorteo);
    intervaloSorteo = null;
    
    cerrarModalSorteo();
    
    // Esperar un momento antes de mostrar resultado
    setTimeout(() => {
        mostrarModalResultado();
    }, 500);
    
    sorteoActivo = false;
}

/**
 * Mostrar modal de sorteo en progreso
 */
function mostrarModalSorteo() {
    const modal = document.getElementById('modalSorteo');
    modal.classList.add('active');
    
    // Botón detener
    document.getElementById('btnDetener').onclick = function() {
        if (confirm('¿Deseas detener el sorteo actual?')) {
            detenerSorteo();
        }
    };
}

/**
 * Cerrar modal de sorteo
 */
function cerrarModalSorteo() {
    const modal = document.getElementById('modalSorteo');
    modal.classList.remove('active');
}

/**
 * Detener sorteo manualmente
 */
function detenerSorteo() {
    clearInterval(intervaloSorteo);
    intervaloSorteo = null;
    sorteoActivo = false;
    cerrarModalSorteo();
}

/**
 * Mostrar modal de resultado final
 */
function mostrarModalResultado() {
    if (!sorteoData) return;
    
    const modal = document.getElementById('modalResultado');
    const numeroGanadorFinal = document.getElementById('numeroGanadorFinal');
    const repeticionesCompletadas = document.getElementById('repeticionesCompletadas');
    const totalSorteos = document.getElementById('totalSorteos');
    const distribucionFinal = document.getElementById('distribucionFinal');
    
    // Llenar datos
    numeroGanadorFinal.textContent = sorteoData.ganador;
    
    const ultimoSorteo = sorteoData.sorteos[sorteoData.sorteos.length - 1];
    repeticionesCompletadas.textContent = ultimoSorteo.contador_ganador;
    totalSorteos.textContent = sorteoData.total_sorteos;
    
    // Mostrar distribución
    if (sorteoData.estadisticas) {
        const stats = sorteoData.estadisticas;
        distribucionFinal.innerHTML = `
            <p><span>Total de números:</span> <strong>${stats.total_numeros}</strong></p>
            <p><span>Números con 1 vez:</span> <strong>${stats.con_1_vez}</strong></p>
            <p><span>Números con 2 veces:</span> <strong>${stats.con_2_veces}</strong></p>
            <p><span>Números con 3 veces:</span> <strong>${stats.con_3_veces}</strong></p>
            <p><span>Números con 4 veces:</span> <strong>${stats.con_4_veces}</strong></p>
            <p><span>Números con 5+ veces:</span> <strong>${stats.con_5_o_mas}</strong></p>
            <hr style="margin: 10px 0; border: none; border-top: 1px solid rgba(255,255,255,0.1);">
            <p><span>Números con 2+ veces:</span> <strong>${stats.porcentaje_con_2_o_mas}%</strong></p>
        `;
    }
    
    modal.classList.add('active');
    
    // Botones
    document.getElementById('btnNuevoSorteo').onclick = function() {
        modal.classList.remove('active');
        window.location.reload();
    };
    
    document.getElementById('btnVerDetalles').onclick = function() {
        mostrarDetallesSorteo();
    };
}

/**
 * Mostrar detalles completos del sorteo
 */
function mostrarDetallesSorteo() {
    if (!sorteoData) return;
    
    let html = '<div style="max-height: 400px; overflow-y: auto; font-size: 12px;">';
    html += '<h4 style="margin-bottom: 10px;">Secuencia Completa:</h4>';
    
    sorteoData.sorteos.forEach(sorteo => {
        const emoji = sorteo.es_ganador ? '⭐' : '';
        const clase = sorteo.es_ganador ? 'style="color: #4CAF50; font-weight: bold;"' : '';
        html += `<p ${clase}>Sorteo #${sorteo.sorteo}: ${sorteo.numero} ${emoji} (Apariciones: ${sorteo.apariciones})</p>`;
    });
    
    html += '</div>';
    
    const modal = document.createElement('div');
    modal.className = 'modal active';
    modal.innerHTML = `
        <div class="modal-content">
            <div class="modal-header">
                <h3>Detalles del Sorteo</h3>
            </div>
            <div class="modal-body">
                ${html}
            </div>
            <div class="modal-footer">
                <button onclick="this.closest('.modal').remove()" class="btn-nuevo-sorteo">Cerrar</button>
            </div>
        </div>
    `;
    
    document.body.appendChild(modal);
}

/**
 * Cerrar modal de resultado
 */
function cerrarModalResultado() {
    const modal = document.getElementById('modalResultado');
    modal.classList.remove('active');
}
