<?php
/**
 * BOLILLERO - Lógica ULTRA CONFIABLE
 * El número de BD SIEMPRE gana primero, de forma IMPERCEPTIBLE
 */

class SorteoLogic {
    
    private $db;
    
    public function __construct() {
        $this->db = getDB();
    }
    
    public function obtenerGanadorActivo() {
        $stmt = $this->db->prepare("
            SELECT numero_ganador, repeticiones_necesarias 
            FROM configuracion_ganador 
            WHERE esta_activo = 1 
            LIMIT 1
        ");
        $stmt->execute();
        return $stmt->fetch();
    }
    
    public function validarConfiguracion($desde, $hasta, $repeticiones) {
        $errores = [];
        
        if ($desde < 0 || $hasta < 0) {
            $errores[] = "Los números deben ser positivos";
        }
        
        if ($desde >= $hasta) {
            $errores[] = "El número 'hasta' debe ser mayor que 'desde'";
        }
        
        if ($repeticiones < 1 || $repeticiones > 10) {
            $errores[] = "Las repeticiones deben estar entre 1 y 10";
        }
        
        $rango = $hasta - $desde + 1;
        if ($rango < 5) {
            $errores[] = "El rango debe tener al menos 5 números";
        }
        
        return $errores;
    }
    
    /**
     * ========================================
     * ALGORITMO PRINCIPAL - 100% CONFIABLE
     * ========================================
     * GARANTIZA que el ganador de BD sea el primero
     * De manera TOTALMENTE IMPERCEPTIBLE
     */
    public function obtenerProximoNumero($desde, $hasta, $repetir, $numero_ganador_config, $contador_sorteos, $contador, $primer_ganador, $ganadores = []) {
        
        // ===== NÚMEROS DISPONIBLES (SIN GANADORES) =====
        $numeros_disponibles = [];
        for ($i = $desde; $i <= $hasta; $i++) {
            if (!in_array($i, $ganadores)) {
                $numeros_disponibles[] = $i;
            }
        }
        
        if (empty($numeros_disponibles)) {
            return rand($desde, $hasta);
        }
        
        if (count($numeros_disponibles) == 1) {
            return $numeros_disponibles[0];
        }

        // ===== VERIFICACIÓN CRÍTICA: SI YA HAY PRIMER GANADOR =====
        if ($primer_ganador !== null) {
            // Si el ganador config NO fue el primero, continuar sorteo normal
            return $this->sorteoNormalSinControl($numeros_disponibles, $contador);
        }

        // ===== NO HAY GANADOR AÚN - FASE DE CONTROL =====
        
        // Si no hay ganador configurado o está fuera de rango, sorteo normal
        if ($numero_ganador_config === null || !in_array($numero_ganador_config, $numeros_disponibles)) {
            return $this->sorteoNormalSinControl($numeros_disponibles, $contador);
        }

        // ===== ESTADÍSTICAS =====
        $contador_ganador = $contador[$numero_ganador_config];
        $min_apariciones = PHP_INT_MAX;
        $max_apariciones = 0;
        $suma_apariciones = 0;
        
        foreach ($numeros_disponibles as $num) {
            $cant = $contador[$num];
            if ($cant < $min_apariciones) $min_apariciones = $cant;
            if ($cant > $max_apariciones) $max_apariciones = $cant;
            $suma_apariciones += $cant;
        }
        
        $promedio = $suma_apariciones / count($numeros_disponibles);
        
        // ===== ESTRATEGIA 1: REGLA DEL 90% =====
        // El 90% debe estar a 1 paso de ganar antes de permitir ganadores
        $umbral_critico = $repetir - 1; // Si repetir=5, umbral=4
        
        $count_en_umbral = 0;
        foreach ($numeros_disponibles as $num) {
            if ($contador[$num] >= $umbral_critico) {
                $count_en_umbral++;
            }
        }
        
        $porcentaje_en_umbral = ($count_en_umbral / count($numeros_disponibles)) * 100;
        
        // ===== FASE 1: ANTES DEL 90% =====
        if ($porcentaje_en_umbral < 70) {
            
            // Calcular distancia del ganador respecto al promedio
            $distancia_del_promedio = $promedio - $contador_ganador;
            
            // Si el ganador está MUY por debajo del promedio, FORZARLO
            if ($distancia_del_promedio >= 1.5) {
                if (rand(1, 100) <= 85) { // 85% de probabilidad
                    return $numero_ganador_config;
                }
            }
            // Si está algo por debajo
            elseif ($distancia_del_promedio >= 0.8) {
                if (rand(1, 100) <= 65) {
                    return $numero_ganador_config;
                }
            }
            // Si está cerca del promedio
            elseif ($distancia_del_promedio >= 0) {
                if (rand(1, 100) <= 45) {
                    return $numero_ganador_config;
                }
            }
            
            // Si NO salió el ganador, elegir otros EVITANDO que alcancen el umbral
            $candidatos = [];
            foreach ($numeros_disponibles as $num) {
                if ($num != $numero_ganador_config && $contador[$num] < $umbral_critico) {
                    $candidatos[] = $num;
                }
            }
            
            if (!empty($candidatos)) {
                return $this->seleccionarConPesoEquilibrado($candidatos, $contador);
            }
            
            // Si todos están en umbral menos el ganador, FORZAR ganador
            if ($contador_ganador < $umbral_critico) {
                return $numero_ganador_config;
            }
            
            // Último recurso
            return $this->seleccionarConPesoEquilibrado($numeros_disponibles, $contador);
        }

        // ===== FASE 2: YA SE ALCANZÓ 90%+ EN UMBRAL =====
        // Ahora SÍ se permite que ganen, pero SOLO el ganador config primero
        
        // VERIFICACIÓN CRÍTICA: ¿El ganador config ya está listo para ganar?
        if ($contador_ganador >= $repetir) {
            // SI YA TIENE LAS REPETICIONES, FORZARLO CON 95% DE PROBABILIDAD
            if (rand(1, 100) <= 95) {
                return $numero_ganador_config;
            }
        }
        
        // Si el ganador está a 1 paso de ganar (umbral crítico)
        if ($contador_ganador >= $umbral_critico) {
            // PROBABILIDAD MUY ALTA de que salga
            if (rand(1, 100) <= 80) {
                return $numero_ganador_config;
            }
        }
        
        // Si el ganador está a 2 pasos
        if ($contador_ganador >= ($repetir - 2)) {
            if (rand(1, 100) <= 60) {
                return $numero_ganador_config;
            }
        }
        
        // Calcular distancia respecto al máximo
        $max_otros = 0;
        foreach ($numeros_disponibles as $num) {
            if ($num != $numero_ganador_config && $contador[$num] > $max_otros) {
                $max_otros = $contador[$num];
            }
        }
        
        $diferencia = $max_otros - $contador_ganador;
        
        // Si está muy atrás, aumentar probabilidad
        if ($diferencia >= 2) {
            if (rand(1, 100) <= 70) {
                return $numero_ganador_config;
            }
        } elseif ($diferencia >= 1) {
            if (rand(1, 100) <= 55) {
                return $numero_ganador_config;
            }
        } else {
            // Está empatado o adelante
            if (rand(1, 100) <= 45) {
                return $numero_ganador_config;
            }
        }
        
        // Si NO salió, elegir OTRO número que NO esté listo para ganar
        $otros_seguros = [];
        foreach ($numeros_disponibles as $num) {
            if ($num != $numero_ganador_config && $contador[$num] < $repetir) {
                $otros_seguros[] = $num;
            }
        }
        
        if (!empty($otros_seguros)) {
            return $this->seleccionarConPesoEquilibrado($otros_seguros, $contador);
        }
        
        // Último recurso: el ganador DEBE salir
        return $numero_ganador_config;
    }
    
    /**
     * Sorteo normal SIN control (cuando ya hay ganador o no hay config)
     */
    private function sorteoNormalSinControl($numeros, $contador) {
        return $this->seleccionarConPesoEquilibrado($numeros, $contador);
    }
    
    /**
     * Selección equilibrada (favorece los que tienen menos)
     */
    private function seleccionarConPesoEquilibrado($numeros, $contador) {
        
        if (empty($numeros)) {
            return $numeros[0];
        }
        
        if (count($numeros) == 1) {
            return $numeros[0];
        }
        
        // Obtener apariciones
        $apariciones = [];
        foreach ($numeros as $num) {
            $apariciones[$num] = $contador[$num];
        }
        
        $min = min($apariciones);
        $max = max($apariciones);
        
        // Si todos iguales, aleatorio
        if ($min == $max) {
            return $numeros[array_rand($numeros)];
        }
        
        // Crear pool ponderado (los con menos apariciones tienen más peso)
        $pool = [];
        
        foreach ($apariciones as $num => $cant) {
            // Peso: (max - cant + 1) ^ 2
            $peso = pow(($max - $cant + 1), 2);
            
            for ($i = 0; $i < $peso; $i++) {
                $pool[] = $num;
            }
        }
        
        if (empty($pool)) {
            return $numeros[array_rand($numeros)];
        }
        
        return $pool[array_rand($pool)];
    }
    
    /**
     * Métodos auxiliares
     */
    private function sorteoAleatorio($desde, $hasta, $repeticiones) {
        return [
            'success' => true,
            'mensaje' => 'Sorteo aleatorio',
            'ganador' => rand($desde, $hasta)
        ];
    }
    
    public function ejecutarSorteoControlado($desde, $hasta, $repeticiones) {
        $errores = $this->validarConfiguracion($desde, $hasta, $repeticiones);
        if (!empty($errores)) {
            return ['error' => true, 'mensaje' => implode('. ', $errores)];
        }
        
        $config_ganador = $this->obtenerGanadorActivo();
        if (!$config_ganador) {
            return ['error' => true, 'mensaje' => 'No hay ganador configurado'];
        }
        
        $numero_ganador = (int)$config_ganador['numero_ganador'];
        
        if ($numero_ganador < $desde || $numero_ganador > $hasta) {
            return $this->sorteoAleatorio($desde, $hasta, $repeticiones);
        }
        
        return [
            'success' => true,
            'ganador' => $numero_ganador
        ];
    }
}
?>