<?php
// Mostrar todos los errores solo para desarrollo (quítalo en producción)
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

require_once '../config/config.php';
require_once '../includes/sorteo_logic.php';

header('Content-Type: application/json');

try {
    $input = json_decode(file_get_contents('php://input'), true);

    $desde = isset($input['desde']) ? (int)$input['desde'] : 0;
    $hasta = isset($input['hasta']) ? (int)$input['hasta'] : 0;
    $repetir = isset($input['repetir']) ? (int)$input['repetir'] : 0;

    if ($desde <= 0 || $hasta <= 0 || $repetir <= 0) {
        echo json_encode(['error' => true, 'mensaje' => 'Parámetros inválidos']);
        exit;
    }

    if ($desde >= $hasta) {
        echo json_encode(['error' => true, 'mensaje' => 'El "hasta" debe ser mayor que "desde"']);
        exit;
    }

    $session_id = get_or_create_session_id();
    $db = getDB();

    // Obtener sesión existente
    $stmt = $db->prepare("
        SELECT id, contador_sorteos, numeros_sorteados, ganadores
        FROM sesiones_sorteo 
        WHERE session_id = :session_id 
          AND desde = :desde 
          AND hasta = :hasta 
          AND repetir = :repetir
          AND estado = 'activo'
        LIMIT 1
    ");
    $stmt->execute([
        'session_id' => $session_id,
        'desde' => $desde,
        'hasta' => $hasta,
        'repetir' => $repetir
    ]);
    $sesion = $stmt->fetch();

    if (!$sesion) {
        // Crear nueva sesión
        $stmt = $db->prepare("
            INSERT INTO sesiones_sorteo 
            (session_id, desde, hasta, repetir, contador_sorteos, numeros_sorteados, ganadores, estado)
            VALUES (:session_id, :desde, :hasta, :repetir, 0, '[]', '[]', 'activo')
        ");
        $stmt->execute([
            'session_id' => $session_id,
            'desde' => $desde,
            'hasta' => $hasta,
            'repetir' => $repetir
        ]);

        $sesion_id = $db->lastInsertId();
        $contador_sorteos = 0;
        $numeros_sorteados = [];
        $ganadores = [];
    } else {
        $sesion_id = $sesion['id'];
        $contador_sorteos = (int)$sesion['contador_sorteos'];
        $numeros_sorteados = json_decode($sesion['numeros_sorteados'], true) ?: [];
        $ganadores = json_decode($sesion['ganadores'], true) ?: [];
    }

    // Verificar si el juego terminó
    $total_numeros = $hasta - $desde + 1;
    if (count($ganadores) >= $total_numeros) {
        echo json_encode([
            'success' => true,
            'juego_terminado' => true,
            'mensaje' => '¡Todos los números han ganado! Juego completado.',
            'ganadores' => $ganadores
        ]);

        $stmt = $db->prepare("UPDATE sesiones_sorteo SET estado = 'completado' WHERE id = :id");
        $stmt->execute(['id' => $sesion_id]);
        exit;
    }

    // Contar apariciones (solo números no ganadores)
    $contador = array_fill_keys(range($desde, $hasta), 0);
    foreach ($numeros_sorteados as $num) {
        if (isset($contador[$num])) $contador[$num]++;
    }

    // Obtener ganador activo de la BD
    $sorteoLogic = new SorteoLogic();
    $config_ganador = $sorteoLogic->obtenerGanadorActivo();
    $numero_ganador_config = null;
    if ($config_ganador) {
        $numero_ganador_config = (int)$config_ganador['numero_ganador'];
        if ($numero_ganador_config < $desde || $numero_ganador_config > $hasta) {
            $numero_ganador_config = null;
        }
    }

    // Primer ganador en la sesión (para lógica interna)
    $primer_ganador = count($ganadores) > 0 ? $ganadores[0] : null;

    // ===== OBTENER NÚMERO =====
    $numero_sorteado = $sorteoLogic->obtenerProximoNumero(
        $desde,
        $hasta,
        $repetir,
        $numero_ganador_config,
        $contador_sorteos,
        $contador,
        $primer_ganador,
        $ganadores
    );

    // Actualizar
    $contador_sorteos++;
    $numeros_sorteados[] = $numero_sorteado;
    $contador[$numero_sorteado]++;

    $es_nuevo_ganador = false;
    if ($contador[$numero_sorteado] >= $repetir && !in_array($numero_sorteado, $ganadores)) {
        $ganadores[] = $numero_sorteado;
        $es_nuevo_ganador = true;
    }

    // Guardar sesión
    $stmt = $db->prepare("
        UPDATE sesiones_sorteo
        SET contador_sorteos = :contador,
            numeros_sorteados = :numeros,
            ganadores = :ganadores,
            ultima_actualizacion = NOW()
        WHERE id = :id
    ");
    $stmt->execute([
        'contador' => $contador_sorteos,
        'numeros' => json_encode($numeros_sorteados),
        'ganadores' => json_encode($ganadores),
        'id' => $sesion_id
    ]);

    echo json_encode([
        'success' => true,
        'numero' => $numero_sorteado,
        'sorteo_actual' => $contador_sorteos,
        'apariciones' => $contador[$numero_sorteado],
        'es_nuevo_ganador' => $es_nuevo_ganador,
        'ganadores' => $ganadores,
        'total_ganadores' => count($ganadores),
        'numeros_restantes' => $total_numeros - count($ganadores),
        'juego_terminado' => false
    ]);

} catch (Exception $e) {
    echo json_encode([
        'error' => true,
        'mensaje' => 'Error: ' . $e->getMessage()
    ]);
}
?>
